// Flappy Bird Game
// This is a fully working single-page game with no build step

// Game canvas setup
const canvas = document.getElementById('gameCanvas');
const ctx = canvas.getContext('2d');

// Game constants
const GRAVITY = 0.5;
const JUMP_FORCE = -10;
const PIPE_SPEED = 3;
const PIPE_GAP = 150;
const PIPE_FREQUENCY = 180; // Frames between pipe spawns

// Game state
let gameRunning = false;
let score = 0;
let frameCount = 0;

// Bird object
const bird = {
    x: 100,
    y: 300,
    width: 34,
    height: 24,
    velocity: 0,
    draw: function() {
        ctx.fillStyle = '#ffff00';
        ctx.fillRect(this.x, this.y, this.width, this.height);

        // Draw bird eyes
        ctx.fillStyle = '#000';
        ctx.beginPath();
        ctx.arc(this.x + 20, this.y + 8, 4, 0, Math.PI * 2);
        ctx.fill();
        
        // Draw bird beak
        ctx.fillStyle = '#ff6600';
        ctx.beginPath();
        ctx.moveTo(this.x + 34, this.y + 12);
        ctx.lineTo(this.x + 40, this.y + 12);
        ctx.lineTo(this.x + 34, this.y + 16);
        ctx.closePath();
        ctx.fill();
    },
    update: function() {
        // Apply gravity
        this.velocity += GRAVITY;
        this.y += this.velocity;

        // Prevent bird from going off the top or bottom
        if (this.y < 0) {
            this.y = 0;
            this.velocity = 0;
        }
        if (this.y + this.height > canvas.height) {
            this.y = canvas.height - this.height;
            this.velocity = 0;
            gameOver();
        }
    }
};

// Pipe object
const pipes = [];

function createPipe() {
    const minHeight = 50;
    const maxHeight = canvas.height - PIPE_GAP - minHeight;
    const height = Math.floor(Math.random() * (maxHeight - minHeight + 1)) + minHeight;

    const pipe = {
        x: canvas.width,
        topHeight: height,
        bottomY: height + PIPE_GAP,
        passed: false,
        draw: function() {
            // Top pipe
            ctx.fillStyle = '#73ad26';
            ctx.fillRect(this.x, 0, 52, this.topHeight);

            // Green pipe cap
            ctx.fillStyle = '#689f38';
            ctx.fillRect(this.x - 5, this.topHeight - 15, 62, 15);

            // Bottom pipe
            ctx.fillStyle = '#73ad26';
            ctx.fillRect(this.x, this.bottomY, 52, canvas.height - this.bottomY);

            // Green pipe cap
            ctx.fillStyle = '#689f38';
            ctx.fillRect(this.x - 5, this.bottomY, 62, 15);
        },
        update: function() {
            this.x -= PIPE_SPEED;

            // Check if bird passed the pipe
            if (!this.passed && this.x + 52 < bird.x) {
                this.passed = true;
                score++;
                document.getElementById('score').textContent = score;
            }

            // Remove pipe when it goes off-screen
            if (this.x + 52 < 0) {
                pipes.splice(pipes.indexOf(this), 1);
            }
        }
    };

    pipes.push(pipe);
}

// Game over function
function gameOver() {
    gameRunning = false;
    document.getElementById('restartButton').style.display = 'block';
}

// Restart game function
function restartGame() {
    bird.y = 300;
    bird.velocity = 0;
    pipes.length = 0;
    score = 0;
    document.getElementById('score').textContent = score;
    document.getElementById('restartButton').style.display = 'none';
    gameRunning = true;
    frameCount = 0;
    gameLoop();
}

// Handle keyboard input
document.addEventListener('keydown', function(e) {
    if (e.code === 'Space' && gameRunning) {
        bird.velocity = JUMP_FORCE;
    } else if (e.code === 'Space' && !gameRunning) {
        restartGame();
    }
});

// Collision detection
function checkCollision() {
    // Check collision with top and bottom of canvas
    if (bird.y <= 0 || bird.y + bird.height >= canvas.height) {
        return true;
    }

    // Check collision with pipes
    for (let i = 0; i < pipes.length; i++) {
        const pipe = pipes[i];

        // Check if bird is within the horizontal bounds of the pipe
        if (bird.x + bird.width > pipe.x && bird.x < pipe.x + 52) {
            // Check collision with top pipe
            if (bird.y < pipe.topHeight) {
                return true;
            }
            
            // Check collision with bottom pipe
            if (bird.y + bird.height > pipe.bottomY) {
                return true;
            }
        }
    }

    return false;
}

// Game loop
function gameLoop() {
    if (!gameRunning) return;

    // Clear canvas
    ctx.clearRect(0, 0, canvas.width, canvas.height);

    // Update game elements
    bird.update();
    for (let i = 0; i < pipes.length; i++) {
        pipes[i].update();
    }

    // Create new pipes
    frameCount++;
    if (frameCount % PIPE_FREQUENCY === 0) {
        createPipe();
    }

    // Check for collisions
    if (checkCollision()) {
        gameOver();
    }

    // Draw game elements
    bird.draw();
    for (let i = 0; i < pipes.length; i++) {
        pipes[i].draw();
    }

    // Continue the game loop
    requestAnimationFrame(gameLoop);
}

// Initialize game
function initGame() {
    // Set up restart button
    const restartButton = document.getElementById('restartButton');
    restartButton.addEventListener('click', restartGame);

    // Start game loop
    gameLoop();
}

// Start the game when page loads
window.addEventListener('load', initGame);
